﻿using System;
using System.Configuration;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.Text;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Schema;
using System.Xml.Xsl;

namespace Ofsc
{
    [ServiceBehavior(Namespace = "http://www.ofsc.org/namespace/Device/2008/11", InstanceContextMode = InstanceContextMode.Single)]
    public class OFSCDeviceService : IOFSCDeviceService, IDisposable
    {
        // デバイス名と下層サービスエンドポイントアドレスの変換テーブル
        private Dictionary<string, string> endpoint = new Dictionary<string, string>()
        {
            // TM-T88V-i
            { "kitchen1", ConfigurationManager.AppSettings["Kitchen1"] },
            // FVP10
            { "kitchen2", "http://localhost:8082/wspos" },
            // CTS801
            { "kitchen3", "http://localhost:8083/wspos" },
            // TM-T90KP
            { "kitchen4", "http://localhost:8084/wspos" },
            // FVP10
            { "kitchen5", "http://localhost:8085/wspos" },
            // CTS801
            { "kitchen6", "http://localhost:8086/wspos" },
        };

        // XML名前空間
        private XNamespace soap = "http://schemas.xmlsoap.org/soap/envelope/";
        private XNamespace ofsc = "http://www.ofsc.org/namespace/Device/2008/11";
        private XNamespace upos = "http://www.nrf-arts.org/UnifiedPOS/POSPrinter/";

        #region OFSC Service

        // OFSC サービス
        public XElement DeviceExecute(XElement request)
        {
            // 実行結果
            var result = true;
            // 実行結果の詳細情報
            var description = "";
            // 応答メッセージ
            XElement response = XElement.Parse(@"
                <s:Envelope xmlns:s='http://schemas.xmlsoap.org/soap/envelope/'>
                    <s:Body>
                        <DeviceExecuteResponse xmlns='http://www.ofsc.org/namespace/Device/2008/11' MajorVersion='1' MinorVersion='0' FixVersion='0'>
                            <ARTSHeader>
                                <MessageID/>
                                <Response ResponseCode='OK'>
                                    <RequestID/>
                                </Response>
                                <ResponseDescription/>
                            </ARTSHeader>
                        </DeviceExecuteResponse>
                    </s:Body>
                </s:Envelope>
            ");
            // 応答メッセージ本文
            XElement res = response.Descendants(ofsc + "DeviceExecuteResponse").First();
            // 応答メッセージにメッセージIDを追加する
            res.Descendants(ofsc + "MessageID").First().Value = Guid.NewGuid().ToString();

            try
            {
                // 要求メッセージ本文
                XElement req = request.Descendants(ofsc + "DeviceExecuteRequest").First();

                // 応答メッセージに要求メッセージIDを追加する
                string mid = req.Descendants(ofsc + "MessageID").First().Value;
                res.Descendants(ofsc + "RequestID").First().Value = (mid == null) ? "" : mid;

                // Device要素の集合を取得する
                var devices = req.Elements(ofsc + "Device");

                // 各Device要素について処理する
                foreach (XElement reqdev in devices)
                {
                    XElement resdev = new XElement(ofsc + "Device");
                    result &= Execute(request, reqdev, resdev);
                    res.Add(resdev);
                }
            }
            catch (Exception e)
            {
                // 実行失敗
                result = false;
                description = e.Message;
            }

            // 応答メッセージに実行結果を追加する
            res.Descendants(ofsc + "Response").First().Attribute("ResponseCode").Value = result ? "OK" : "Rejected";
            res.Descendants(ofsc + "ResponseDescription").First().Value = description;

            // 応答メッセージを返す
            return response;
        }

        #endregion

        #region OFSC Device

        // Device要素を処理する
        private bool Execute(XElement request, XElement reqdev, XElement resdev)
        {
            // 実行結果
            bool result = false;
            // 実行対象デバイス名
            string name = "";
            // 使用するスタイルシート
            string stylesheet = "";
            // タイムアウト
            string timeout = "";
            // 実行結果の詳細情報
            string description = "";

            try
            {
                // Device要素の子要素を取り出す
                name = reqdev.Element(ofsc + "Name").Value;
                stylesheet = reqdev.Element(ofsc + "Stylesheet").Value;
                timeout = reqdev.Element(ofsc + "Timeout").Value;

                // TM-T88V-i OFSC-Print を呼び出す
                if (name == "kitchen1")
                {
                    // Device要素を置き換える
                    XElement reqws = new XElement(request);
                    reqws.Descendants(ofsc + "Device").Remove();
                    reqws.Descendants(ofsc + "ARTSHeader").First().AddAfterSelf(new XElement(ofsc + "Device",
                        new XElement(ofsc + "Name", "local_printer"),
                        new XElement(ofsc + "Stylesheet", stylesheet),
                        new XElement(ofsc + "Timeout", timeout)));

                    // メッセージを転送する
                    WebClient client = new WebClient();
                    client.Encoding = Encoding.UTF8;
                    client.Headers.Set("Content-Type", "text/xml; charset=utf-8");
                    client.Headers.Set("SOAPAction", "\"\"");

                    // 応答メッセージを受信する
                    XElement resws = XElement.Parse(client.UploadString(endpoint[name], reqws.ToString()));
                    result = (resws.Descendants(ofsc + "Response").First().Attribute("ResponseCode").Value == "OK");
                }

                // WS-POS 1.1 Service を呼び出す (Command Set)
                else
                {
                    // XSL変換
                    XDocument doc = new XDocument();
                    using (XmlWriter writer = doc.CreateWriter())
                    {
                        XslCompiledTransform xslt = new XslCompiledTransform(true);
                        xslt.Load(stylesheet);
                        xslt.Transform(request.CreateReader(), writer);
                    }
                    XElement reqws = doc.Root;

                    // 変換後のメッセージにメッセージIDを追加する
                    reqws.Descendants(upos + "MessageID").First().Value = Guid.NewGuid().ToString();

                    // 変換後のメッセージを送信する
                    WebClient client = new WebClient();
                    client.Encoding = Encoding.UTF8;
                    client.Headers.Set("Content-Type", "text/xml; charset=utf-8");
                    client.Headers.Set("SOAPAction", "\"http://www.nrf-arts.org/UnifiedPOS/POSPrinter/POSPrinter\"");

                    // 応答メッセージを受信する
                    XElement resws = XElement.Parse(client.UploadString(endpoint[name], reqws.ToString()));
                    result = (resws.Descendants(upos + "Response").First().Attribute("ResponseCode").Value == "OK");
                }
            }
            catch (Exception e)
            {
                // 実行失敗
                description = e.Message;
            }

            // 応答メッセージに実行結果を追加する
            resdev.Add(new XElement(ofsc + "Name", name));
            resdev.Add(new XElement(ofsc + "Response",
                new XAttribute("ResponseCode", result ? "OK" : "Rejected"),
                new XElement(ofsc + "RequestID"),
                new XElement(ofsc + "ResponseDescription", description)));

            // 実行成功の場合
            if (result)
            {
                // 逐次処理
                XElement reqdevok = reqdev.Element(ofsc + "DeviceOK");
                if (reqdevok != null)
                {
                    // 再帰
                    XElement resdevok = new XElement(ofsc + "DeviceOK");
                    result = Execute(request, reqdevok, resdevok);
                    resdev.Add(resdevok);
                }
            }

            // 実行失敗の場合
            else
            {
                // 迂回処理
                XElement reqdevrej = reqdev.Element(ofsc + "DeviceRejected");
                if (reqdevrej != null)
                {
                    // 再帰
                    XElement resdevrej = new XElement(ofsc + "DeviceRejected");
                    result = Execute(request, reqdevrej, resdevrej);
                    resdev.Add(resdevrej);
                }
            }

            // 実行結果を返す
            return result;
        }

        #endregion

        #region OFSC Dispose

        public void Dispose()
        {
        }

        #endregion
    }
}