﻿using System;
using System.Configuration;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.Text;
using System.Text.RegularExpressions;

namespace Wspos
{
    [ServiceBehavior(Namespace = "http://www.nrf-arts.org/UnifiedPOS/POSPrinter/", InstanceContextMode = InstanceContextMode.Single)]
    public class POSPrinterService : POSPrinterCommandSet, IDisposable
    {
        private Microsoft.PointOfService.PosPrinter device;

        #region POSPrinter Enumration Converter

        // POS for .NETの列挙値とWS-POSの列挙値の変換テーブル
        private static Dictionary<Enum, Enum> Convert = new Dictionary<Enum, Enum>()
            {
                { Microsoft.PointOfService.PowerReporting.Advanced, PowerReporting.Advanced },
                { Microsoft.PointOfService.PowerReporting.None, PowerReporting.None },
                { Microsoft.PointOfService.PowerReporting.Standard, PowerReporting.Standard },
                { PowerReporting.Advanced, Microsoft.PointOfService.PowerReporting.Advanced },
                { PowerReporting.None, Microsoft.PointOfService.PowerReporting.None },
                { PowerReporting.Standard, Microsoft.PointOfService.PowerReporting.Standard },

                { Microsoft.PointOfService.PowerNotification.Disabled, PowerNotification.Disabled },
                { Microsoft.PointOfService.PowerNotification.Enabled, PowerNotification.Enabled },
                { PowerNotification.Disabled, Microsoft.PointOfService.PowerNotification.Disabled },
                { PowerNotification.Enabled, Microsoft.PointOfService.PowerNotification.Enabled },

                { Microsoft.PointOfService.PowerState.Off, PowerState.Off },
                { Microsoft.PointOfService.PowerState.Offline, PowerState.Offline },
                { Microsoft.PointOfService.PowerState.OffOffline, PowerState.OffOffline },
                { Microsoft.PointOfService.PowerState.Online, PowerState.Online },
                { Microsoft.PointOfService.PowerState.Unknown, PowerState.Unknown },
                { PowerState.Off, Microsoft.PointOfService.PowerState.Off },
                { PowerState.Offline, Microsoft.PointOfService.PowerState.Offline },
                { PowerState.OffOffline, Microsoft.PointOfService.PowerState.OffOffline },
                { PowerState.Online, Microsoft.PointOfService.PowerState.Online },
                { PowerState.Unknown, Microsoft.PointOfService.PowerState.Unknown },

                { Microsoft.PointOfService.ControlState.Busy, ControlState.Busy },
                { Microsoft.PointOfService.ControlState.Closed, ControlState.Closed },
                { Microsoft.PointOfService.ControlState.Error, ControlState.Error },
                { Microsoft.PointOfService.ControlState.Idle, ControlState.Idle },
                { ControlState.Busy, Microsoft.PointOfService.ControlState.Busy },
                { ControlState.Closed, Microsoft.PointOfService.ControlState.Closed },
                { ControlState.Error, Microsoft.PointOfService.ControlState.Error },
                { ControlState.Idle, Microsoft.PointOfService.ControlState.Idle },

                { Microsoft.PointOfService.HealthCheckLevel.External, HealthCheckLevel.External },
                { Microsoft.PointOfService.HealthCheckLevel.Interactive, HealthCheckLevel.Interactive },
                { Microsoft.PointOfService.HealthCheckLevel.Internal, HealthCheckLevel.Internal },
                { HealthCheckLevel.External, Microsoft.PointOfService.HealthCheckLevel.External },
                { HealthCheckLevel.Interactive, Microsoft.PointOfService.HealthCheckLevel.Interactive },
                { HealthCheckLevel.Internal, Microsoft.PointOfService.HealthCheckLevel.Internal },

                { Microsoft.PointOfService.CompareFirmwareResult.Different, CompareFirmwareResult.Different },
                { Microsoft.PointOfService.CompareFirmwareResult.Newer, CompareFirmwareResult.Newer },
                { Microsoft.PointOfService.CompareFirmwareResult.Older, CompareFirmwareResult.Older },
                { Microsoft.PointOfService.CompareFirmwareResult.Same, CompareFirmwareResult.Same },
                { Microsoft.PointOfService.CompareFirmwareResult.Unknown, CompareFirmwareResult.Unknown },
                { CompareFirmwareResult.Different, Microsoft.PointOfService.CompareFirmwareResult.Different },
                { CompareFirmwareResult.Newer, Microsoft.PointOfService.CompareFirmwareResult.Newer },
                { CompareFirmwareResult.Older, Microsoft.PointOfService.CompareFirmwareResult.Older },
                { CompareFirmwareResult.Same, Microsoft.PointOfService.CompareFirmwareResult.Same },
                { CompareFirmwareResult.Unknown, Microsoft.PointOfService.CompareFirmwareResult.Unknown },

                { Microsoft.PointOfService.ErrorCode.Busy, ErrorCode.Busy },
                { Microsoft.PointOfService.ErrorCode.Claimed, ErrorCode.Claimed },
                { Microsoft.PointOfService.ErrorCode.Closed, ErrorCode.Closed },
                { Microsoft.PointOfService.ErrorCode.Deprecated, ErrorCode.Deprecated },
                { Microsoft.PointOfService.ErrorCode.Disabled, ErrorCode.Disabled },
                { Microsoft.PointOfService.ErrorCode.Exists, ErrorCode.Exists },
                { Microsoft.PointOfService.ErrorCode.Extended, ErrorCode.Extended },
                { Microsoft.PointOfService.ErrorCode.Failure, ErrorCode.Failure },
                { Microsoft.PointOfService.ErrorCode.Illegal, ErrorCode.Illegal },
                { Microsoft.PointOfService.ErrorCode.NoExist, ErrorCode.NoExist },
                { Microsoft.PointOfService.ErrorCode.NoHardware, ErrorCode.NoHardware },
                { Microsoft.PointOfService.ErrorCode.NoService, ErrorCode.NoService },
                { Microsoft.PointOfService.ErrorCode.NotClaimed, ErrorCode.NotClaimed },
                { Microsoft.PointOfService.ErrorCode.Offline, ErrorCode.Offline },
                { Microsoft.PointOfService.ErrorCode.Success, ErrorCode.Success },
                { Microsoft.PointOfService.ErrorCode.Timeout, ErrorCode.Timeout },
                { ErrorCode.Busy, Microsoft.PointOfService.ErrorCode.Busy },
                { ErrorCode.Claimed, Microsoft.PointOfService.ErrorCode.Claimed },
                { ErrorCode.Closed, Microsoft.PointOfService.ErrorCode.Closed },
                { ErrorCode.Deprecated, Microsoft.PointOfService.ErrorCode.Deprecated },
                { ErrorCode.Disabled, Microsoft.PointOfService.ErrorCode.Disabled },
                { ErrorCode.Exists, Microsoft.PointOfService.ErrorCode.Exists },
                { ErrorCode.Extended, Microsoft.PointOfService.ErrorCode.Extended },
                { ErrorCode.Failure, Microsoft.PointOfService.ErrorCode.Failure },
                { ErrorCode.Illegal, Microsoft.PointOfService.ErrorCode.Illegal },
                { ErrorCode.NoExist, Microsoft.PointOfService.ErrorCode.NoExist },
                { ErrorCode.NoHardware, Microsoft.PointOfService.ErrorCode.NoHardware },
                { ErrorCode.NoService, Microsoft.PointOfService.ErrorCode.NoService },
                { ErrorCode.NotClaimed, Microsoft.PointOfService.ErrorCode.NotClaimed },
                { ErrorCode.Offline, Microsoft.PointOfService.ErrorCode.Offline },
                { ErrorCode.Success, Microsoft.PointOfService.ErrorCode.Success },
                { ErrorCode.Timeout, Microsoft.PointOfService.ErrorCode.Timeout },

                { Microsoft.PointOfService.CharacterSetCapability.Alpha, CharacterSetCapability.Alpha },
                { Microsoft.PointOfService.CharacterSetCapability.Ascii, CharacterSetCapability.Ascii },
                { Microsoft.PointOfService.CharacterSetCapability.Kana, CharacterSetCapability.Kana },
                { Microsoft.PointOfService.CharacterSetCapability.Kanji, CharacterSetCapability.Kanji },
                { Microsoft.PointOfService.CharacterSetCapability.Unicode, CharacterSetCapability.Unicode },
                { CharacterSetCapability.Alpha, Microsoft.PointOfService.CharacterSetCapability.Alpha },
                { CharacterSetCapability.Ascii, Microsoft.PointOfService.CharacterSetCapability.Ascii },
                { CharacterSetCapability.Kana, Microsoft.PointOfService.CharacterSetCapability.Kana },
                { CharacterSetCapability.Kanji, Microsoft.PointOfService.CharacterSetCapability.Kanji },
                { CharacterSetCapability.Unicode, Microsoft.PointOfService.CharacterSetCapability.Unicode },

                { Microsoft.PointOfService.PrinterCartridgeNotify.Disabled, PrinterCartridgeNotify.Disabled },
                { Microsoft.PointOfService.PrinterCartridgeNotify.Enabled, PrinterCartridgeNotify.Enabled },
                { PrinterCartridgeNotify.Disabled, Microsoft.PointOfService.PrinterCartridgeNotify.Disabled },
                { PrinterCartridgeNotify.Enabled, Microsoft.PointOfService.PrinterCartridgeNotify.Enabled },

                { Microsoft.PointOfService.PrinterErrorLevel.Fatal, PrinterErrorLevel.Fatal },
                { Microsoft.PointOfService.PrinterErrorLevel.None, PrinterErrorLevel.None },
                { Microsoft.PointOfService.PrinterErrorLevel.Recoverable, PrinterErrorLevel.Recoverable },
                { PrinterErrorLevel.Fatal, Microsoft.PointOfService.PrinterErrorLevel.Fatal },
                { PrinterErrorLevel.None, Microsoft.PointOfService.PrinterErrorLevel.None },
                { PrinterErrorLevel.Recoverable, Microsoft.PointOfService.PrinterErrorLevel.Recoverable },

                { Microsoft.PointOfService.PrinterStation.Journal, PrinterStation.Journal },
                { Microsoft.PointOfService.PrinterStation.None, PrinterStation.None },
                { Microsoft.PointOfService.PrinterStation.Receipt, PrinterStation.Receipt },
                { Microsoft.PointOfService.PrinterStation.Slip, PrinterStation.Slip },
                { Microsoft.PointOfService.PrinterStation.TwoReceiptJournal, PrinterStation.TwoReceiptJournal },
                { Microsoft.PointOfService.PrinterStation.TwoSlipJournal, PrinterStation.TwoSlipJournal },
                { Microsoft.PointOfService.PrinterStation.TwoSlipReceipt, PrinterStation.TwoSlipReceipt },
                { PrinterStation.Journal, Microsoft.PointOfService.PrinterStation.Journal },
                { PrinterStation.None, Microsoft.PointOfService.PrinterStation.None },
                { PrinterStation.Receipt, Microsoft.PointOfService.PrinterStation.Receipt },
                { PrinterStation.Slip, Microsoft.PointOfService.PrinterStation.Slip },
                { PrinterStation.TwoReceiptJournal, Microsoft.PointOfService.PrinterStation.TwoReceiptJournal },
                { PrinterStation.TwoSlipJournal, Microsoft.PointOfService.PrinterStation.TwoSlipJournal },
                { PrinterStation.TwoSlipReceipt, Microsoft.PointOfService.PrinterStation.TwoSlipReceipt },

                { Microsoft.PointOfService.MapMode.Dots, MapMode.Dots },
                { Microsoft.PointOfService.MapMode.English, MapMode.English },
                { Microsoft.PointOfService.MapMode.Metric, MapMode.Metric },
                { Microsoft.PointOfService.MapMode.Twips, MapMode.Twips },
                { MapMode.Dots, Microsoft.PointOfService.MapMode.Dots },
                { MapMode.English, Microsoft.PointOfService.MapMode.English },
                { MapMode.Metric, Microsoft.PointOfService.MapMode.Metric },
                { MapMode.Twips, Microsoft.PointOfService.MapMode.Twips },

                { Microsoft.PointOfService.PageModePrintDirection.BottomToTop, PageModePrintDirection.BottomToTop },
                { Microsoft.PointOfService.PageModePrintDirection.LeftToRight, PageModePrintDirection.LeftToRight },
                { Microsoft.PointOfService.PageModePrintDirection.None, PageModePrintDirection.None },
                { Microsoft.PointOfService.PageModePrintDirection.RightToLeft, PageModePrintDirection.RightToLeft },
                { Microsoft.PointOfService.PageModePrintDirection.TopToBottom, PageModePrintDirection.TopToBottom },
                { PageModePrintDirection.BottomToTop, Microsoft.PointOfService.PageModePrintDirection.BottomToTop },
                { PageModePrintDirection.LeftToRight, Microsoft.PointOfService.PageModePrintDirection.LeftToRight },
                { PageModePrintDirection.None, Microsoft.PointOfService.PageModePrintDirection.None },
                { PageModePrintDirection.RightToLeft, Microsoft.PointOfService.PageModePrintDirection.RightToLeft },
                { PageModePrintDirection.TopToBottom, Microsoft.PointOfService.PageModePrintDirection.TopToBottom },

                { Microsoft.PointOfService.Rotation.Left90, Rotation.Left90 },
                { Microsoft.PointOfService.Rotation.Normal, Rotation.Normal },
                { Microsoft.PointOfService.Rotation.Right90, Rotation.Right90 },
                { Microsoft.PointOfService.Rotation.Rotate180, Rotation.Rotate180 },
                { Rotation.Left90, Microsoft.PointOfService.Rotation.Left90 },
                { Rotation.Normal, Microsoft.PointOfService.Rotation.Normal },
                { Rotation.Right90, Microsoft.PointOfService.Rotation.Right90 },
                { Rotation.Rotate180, Microsoft.PointOfService.Rotation.Rotate180 },

                { Microsoft.PointOfService.PrinterCartridgeStates.Cleaning, PrinterCartridgeStates.Cleaning },
                { Microsoft.PointOfService.PrinterCartridgeStates.Empty, PrinterCartridgeStates.Empty },
                { Microsoft.PointOfService.PrinterCartridgeStates.NearEnd, PrinterCartridgeStates.NearEnd },
                { Microsoft.PointOfService.PrinterCartridgeStates.OK, PrinterCartridgeStates.OK },
                { Microsoft.PointOfService.PrinterCartridgeStates.Removed, PrinterCartridgeStates.Removed },
                { Microsoft.PointOfService.PrinterCartridgeStates.Unknown, PrinterCartridgeStates.Unknown },
                { PrinterCartridgeStates.Cleaning, Microsoft.PointOfService.PrinterCartridgeStates.Cleaning },
                { PrinterCartridgeStates.Empty, Microsoft.PointOfService.PrinterCartridgeStates.Empty },
                { PrinterCartridgeStates.NearEnd, Microsoft.PointOfService.PrinterCartridgeStates.NearEnd },
                { PrinterCartridgeStates.OK, Microsoft.PointOfService.PrinterCartridgeStates.OK },
                { PrinterCartridgeStates.Removed, Microsoft.PointOfService.PrinterCartridgeStates.Removed },
                { PrinterCartridgeStates.Unknown, Microsoft.PointOfService.PrinterCartridgeStates.Unknown },

                { Microsoft.PointOfService.PrinterSide.Opposite, PrinterSide.Opposite },
                { Microsoft.PointOfService.PrinterSide.Side1, PrinterSide.Side1 },
                { Microsoft.PointOfService.PrinterSide.Side2, PrinterSide.Side2 },
                { Microsoft.PointOfService.PrinterSide.Unknown, PrinterSide.Unknown },
                { PrinterSide.Opposite, Microsoft.PointOfService.PrinterSide.Opposite },
                { PrinterSide.Side1, Microsoft.PointOfService.PrinterSide.Side1 },
                { PrinterSide.Side2, Microsoft.PointOfService.PrinterSide.Side2 },
                { PrinterSide.Unknown, Microsoft.PointOfService.PrinterSide.Unknown },

                { Microsoft.PointOfService.PageModePrintControl.Cancel, PageModePrintControl.Cancel },
                { Microsoft.PointOfService.PageModePrintControl.Normal, PageModePrintControl.Normal },
                { Microsoft.PointOfService.PageModePrintControl.PageMode, PageModePrintControl.PageMode },
                { Microsoft.PointOfService.PageModePrintControl.PrintSave, PageModePrintControl.PrintSave },
                { PageModePrintControl.Cancel, Microsoft.PointOfService.PageModePrintControl.Cancel },
                { PageModePrintControl.Normal, Microsoft.PointOfService.PageModePrintControl.Normal },
                { PageModePrintControl.PageMode, Microsoft.PointOfService.PageModePrintControl.PageMode },
                { PageModePrintControl.PrintSave, Microsoft.PointOfService.PageModePrintControl.PrintSave },

                { Microsoft.PointOfService.BarCodeTextPosition.Above, BarCodeTextPosition.Above },
                { Microsoft.PointOfService.BarCodeTextPosition.Below, BarCodeTextPosition.Below },
                { Microsoft.PointOfService.BarCodeTextPosition.None, BarCodeTextPosition.None },
                { BarCodeTextPosition.Above, Microsoft.PointOfService.BarCodeTextPosition.Above },
                { BarCodeTextPosition.Below, Microsoft.PointOfService.BarCodeTextPosition.Below },
                { BarCodeTextPosition.None, Microsoft.PointOfService.BarCodeTextPosition.None },

                { Microsoft.PointOfService.PrinterLogoLocation.Bottom, PrinterLogoLocation.Bottom },
                { Microsoft.PointOfService.PrinterLogoLocation.Top, PrinterLogoLocation.Top },
                { PrinterLogoLocation.Bottom, Microsoft.PointOfService.PrinterLogoLocation.Bottom },
                { PrinterLogoLocation.Top, Microsoft.PointOfService.PrinterLogoLocation.Top },

                { Microsoft.PointOfService.PrinterTransactionControl.Normal, PrinterTransactionControl.Normal },
                { Microsoft.PointOfService.PrinterTransactionControl.Transaction, PrinterTransactionControl.Transaction },
                { PrinterTransactionControl.Normal, Microsoft.PointOfService.PrinterTransactionControl.Normal },
                { PrinterTransactionControl.Transaction, Microsoft.PointOfService.PrinterTransactionControl.Transaction },
            };

        #endregion

        #region POSPrinter Escape Sequence

        // エスケープシーケンスの正規表現
        private Regex pattern = new Regex("\\\\u([0-9a-fA-F]{4})|(\\\\\\\\)|(\\\\n)|(\\\\r)|(\\\\t)|(\\\\f)|(\\\\b)", RegexOptions.Compiled);

        // エスケープシーケンスを処理する
        private string ProcessEscapeSequence(string data)
        {
            Match m = pattern.Match(data);
            StringBuilder s = new StringBuilder();
            int start = 0;
            while (m.Success)
            {
                s.Append(data.Substring(start, m.Index - start));
                if (m.Groups[1].Length > 0)
                {
                    s.Append(System.Convert.ToChar(System.Convert.ToInt32(m.Groups[1].Value, 16)));
                }
                else if (m.Groups[2].Length > 0)
                {
                    s.Append("\\");
                }
                else if (m.Groups[3].Length > 0)
                {
                    s.Append("\n");
                }
                else if (m.Groups[4].Length > 0)
                {
                    s.Append("\r");
                }
                else if (m.Groups[5].Length > 0)
                {
                    s.Append("\t");
                }
                else if (m.Groups[6].Length > 0)
                {
                    s.Append("\f");
                }
                else if (m.Groups[7].Length > 0)
                {
                    s.Append("\b");
                }
                start = m.Index + m.Length;
                m = m.NextMatch();
            }
            s.Append(data.Substring(start));
            return s.ToString();
        }

        #endregion

        #region POSPrinter Constructor

        // コンストラクタ
        public POSPrinterService()
        {
            // PosPrinterオブジェクトを作成する
            Microsoft.PointOfService.PosExplorer posExplorer = new Microsoft.PointOfService.PosExplorer();
            Microsoft.PointOfService.DeviceInfo deviceInfo = posExplorer.GetDevice("PosPrinter", ConfigurationManager.AppSettings["DeviceName"]);
            device = (Microsoft.PointOfService.PosPrinter)posExplorer.CreateInstance(deviceInfo);
            device.Open();
            device.Claim(60000);
            device.DeviceEnabled = true;
        }

        #endregion

        #region POSPrinter Service

        // WS-POS サービス (Command Set)
        public POSPrinterResponse POSPrinter(POSPrinterRequest request)
        {
            // 要求メッセージのメッセージIDを取得する
            string mid = request.POSPrinter.ARTSHeader.MessageID.Value;

            // 応答メッセージを作成する
            POSPrinterResponse response = new POSPrinterResponse(new POSPrinter());
            response.POSPrinter.ARTSHeader = new ARTSCommonHeaderType();
            response.POSPrinter.ARTSHeader.MessageID = new RequestIDCommonData();
            response.POSPrinter.ARTSHeader.MessageID.Value = Guid.NewGuid().ToString();
            response.POSPrinter.ARTSHeader.Response = new ResponseCommonData();
            response.POSPrinter.ARTSHeader.Response.RequestID = (mid == null) ? "" : mid;
            response.POSPrinter.ARTSHeader.Response.ResponseCode = "OK";
            response.POSPrinter.ARTSHeader.Response.ResponseDescription = new ResponseCommonDataResponseDescription();
            response.POSPrinter.ARTSHeader.Response.ResponseDescription.Value = "";

            // オペレーション(メソッド・プロパティ)
            object[] opration = request.POSPrinter.POSPrinterBody;
            // オペレーションの実行結果
            List<object> result = new List<object>();

            try
            {
                // 各オペレーションについて処理する
                foreach (object obj in opration)
                {
                    // Open メソッド
                    if (obj is POSPrinterBaseOpen)
                    {
                        POSPrinterBaseOpen req = (POSPrinterBaseOpen)obj;
                        POSPrinterBaseOpenResponse res = new POSPrinterBaseOpenResponse();
                        //device.Open();
                        result.Add(res);
                    }

                    // Claim メソッド
                    else if (obj is POSPrinterBaseClaim)
                    {
                        POSPrinterBaseClaim req = (POSPrinterBaseClaim)obj;
                        POSPrinterBaseClaimResponse res = new POSPrinterBaseClaimResponse();
                        //device.Claim(req.Timeout);
                        result.Add(res);
                    }

                    // CheckHealth メソッド
                    else if (obj is POSPrinterBaseCheckHealth)
                    {
                        POSPrinterBaseCheckHealth req = (POSPrinterBaseCheckHealth)obj;
                        POSPrinterBaseCheckHealthResponse res = new POSPrinterBaseCheckHealthResponse();
                        device.CheckHealth((Microsoft.PointOfService.HealthCheckLevel)Convert[req.Level]);
                        result.Add(res);
                    }

                    // Release メソッド
                    else if (obj is POSPrinterBaseRelease)
                    {
                        POSPrinterBaseRelease req = (POSPrinterBaseRelease)obj;
                        POSPrinterBaseReleaseResponse res = new POSPrinterBaseReleaseResponse();
                        //device.Release();
                        result.Add(res);
                    }

                    // Close メソッド
                    else if (obj is POSPrinterBaseClose)
                    {
                        POSPrinterBaseClose req = (POSPrinterBaseClose)obj;
                        POSPrinterBaseCloseResponse res = new POSPrinterBaseCloseResponse();
                        //device.Close();
                        result.Add(res);
                    }

                    // PrintNormal メソッド
                    else if (obj is POSPrinterBasePrintNormal)
                    {
                        POSPrinterBasePrintNormal req = (POSPrinterBasePrintNormal)obj;
                        POSPrinterBasePrintNormalResponse res = new POSPrinterBasePrintNormalResponse();
                        device.PrintNormal((Microsoft.PointOfService.PrinterStation)Convert[req.Station], ProcessEscapeSequence(req.Data));
                        result.Add(res);
                    }

                    // PrintBitmap メソッド
                    else if (obj is POSPrinterBasePrintBitmap)
                    {
                        POSPrinterBasePrintBitmap req = (POSPrinterBasePrintBitmap)obj;
                        POSPrinterBasePrintBitmapResponse res = new POSPrinterBasePrintBitmapResponse();
                        device.PrintBitmap((Microsoft.PointOfService.PrinterStation)Convert[req.Station], req.FileName, req.Width, req.Alignment);
                        result.Add(res);
                    }

                    // PrintBarCode メソッド
                    else if (obj is POSPrinterBasePrintBarCode)
                    {
                        POSPrinterBasePrintBarCode req = (POSPrinterBasePrintBarCode)obj;
                        POSPrinterBasePrintBarCodeResponse res = new POSPrinterBasePrintBarCodeResponse();
                        device.PrintBarCode((Microsoft.PointOfService.PrinterStation)Convert[req.Station], ProcessEscapeSequence(req.Data), (Microsoft.PointOfService.BarCodeSymbology)req.Symbology, req.Height, req.Width, req.Alignment, (Microsoft.PointOfService.BarCodeTextPosition)Convert[req.TextPosition]);
                        result.Add(res);
                    }

                    // RotatePrint メソッド
                    else if (obj is POSPrinterBaseRotatePrint)
                    {
                        POSPrinterBaseRotatePrint req = (POSPrinterBaseRotatePrint)obj;
                        POSPrinterBaseRotatePrintResponse res = new POSPrinterBaseRotatePrintResponse();
                        device.RotatePrint((Microsoft.PointOfService.PrinterStation)Convert[req.Station], (Microsoft.PointOfService.PrintRotation)req.Rotation);
                        result.Add(res);
                    }

                    // TransactionPrint メソッド
                    else if (obj is POSPrinterBaseTransactionPrint)
                    {
                        POSPrinterBaseTransactionPrint req = (POSPrinterBaseTransactionPrint)obj;
                        POSPrinterBaseTransactionPrintResponse res = new POSPrinterBaseTransactionPrintResponse();
                        device.TransactionPrint((Microsoft.PointOfService.PrinterStation)Convert[req.Station], (Microsoft.PointOfService.PrinterTransactionControl)Convert[req.Control]);
                        result.Add(res);
                    }

                    // CutPaper メソッド
                    else if (obj is POSPrinterBaseCutPaper)
                    {
                        POSPrinterBaseCutPaper req = (POSPrinterBaseCutPaper)obj;
                        POSPrinterBaseCutPaperResponse res = new POSPrinterBaseCutPaperResponse();
                        device.CutPaper(req.Percentage);
                        result.Add(res);
                    }

                    // DeviceEnabled プロパティ (Setter)
                    else if (obj is POSPrinterBaseSetDeviceEnabled)
                    {
                        POSPrinterBaseSetDeviceEnabled req = (POSPrinterBaseSetDeviceEnabled)obj;
                        POSPrinterBaseSetDeviceEnabledResponse res = new POSPrinterBaseSetDeviceEnabledResponse();
                        //device.DeviceEnabled = req.DeviceEnabled;
                        result.Add(res);
                    }

                    // DeviceEnabled プロパティ (Getter)
                    else if (obj is POSPrinterBaseGetDeviceEnabled)
                    {
                        POSPrinterBaseGetDeviceEnabled req = (POSPrinterBaseGetDeviceEnabled)obj;
                        POSPrinterBaseGetDeviceEnabledResponse res = new POSPrinterBaseGetDeviceEnabledResponse();
                        res.GetDeviceEnabledResult = device.DeviceEnabled;
                        res.GetDeviceEnabledResultSpecified = true;
                        result.Add(res);
                    }

                    // RecLetterQuality プロパティ (Setter)
                    else if (obj is POSPrinterBaseSetRecLetterQuality)
                    {
                        POSPrinterBaseSetRecLetterQuality req = (POSPrinterBaseSetRecLetterQuality)obj;
                        POSPrinterBaseSetRecLetterQualityResponse res = new POSPrinterBaseSetRecLetterQualityResponse();
                        device.RecLetterQuality = req.RecLetterQuality;
                        result.Add(res);
                    }

                    // RecLetterQuality プロパティ (Getter)
                    else if (obj is POSPrinterBaseGetRecLetterQuality)
                    {
                        POSPrinterBaseGetRecLetterQuality req = (POSPrinterBaseGetRecLetterQuality)obj;
                        POSPrinterBaseGetRecLetterQualityResponse res = new POSPrinterBaseGetRecLetterQualityResponse();
                        res.GetRecLetterQualityResult = device.RecLetterQuality;
                        res.GetRecLetterQualityResultSpecified = true;
                        result.Add(res);
                    }

                    // RecLineChars プロパティ (Setter)
                    else if (obj is POSPrinterBaseSetRecLineChars)
                    {
                        POSPrinterBaseSetRecLineChars req = (POSPrinterBaseSetRecLineChars)obj;
                        POSPrinterBaseSetRecLineCharsResponse res = new POSPrinterBaseSetRecLineCharsResponse();
                        device.RecLineChars = req.RecLineChars;
                        result.Add(res);
                    }

                    // RecLineChars プロパティ (Getter)
                    else if (obj is POSPrinterBaseGetRecLineChars)
                    {
                        POSPrinterBaseGetRecLineChars req = (POSPrinterBaseGetRecLineChars)obj;
                        POSPrinterBaseGetRecLineCharsResponse res = new POSPrinterBaseGetRecLineCharsResponse();
                        res.GetRecLineCharsResult = device.RecLineChars;
                        res.GetRecLineCharsResultSpecified = true;
                        result.Add(res);
                    }

                    // RecLineSpacing プロパティ (Setter)
                    else if (obj is POSPrinterBaseSetRecLineSpacing)
                    {
                        POSPrinterBaseSetRecLineSpacing req = (POSPrinterBaseSetRecLineSpacing)obj;
                        POSPrinterBaseSetRecLineSpacingResponse res = new POSPrinterBaseSetRecLineSpacingResponse();
                        device.RecLineSpacing = req.RecLineSpacing;
                        result.Add(res);
                    }

                    // RecLineSpacing プロパティ (Getter)
                    else if (obj is POSPrinterBaseGetRecLineSpacing)
                    {
                        POSPrinterBaseGetRecLineSpacing req = (POSPrinterBaseGetRecLineSpacing)obj;
                        POSPrinterBaseGetRecLineSpacingResponse res = new POSPrinterBaseGetRecLineSpacingResponse();
                        res.GetRecLineSpacingResult = device.RecLineSpacing;
                        res.GetRecLineSpacingResultSpecified = true;
                        result.Add(res);
                    }

                    // MapMode プロパティ (Setter)
                    else if (obj is POSPrinterBaseSetMapMode)
                    {
                        POSPrinterBaseSetMapMode req = (POSPrinterBaseSetMapMode)obj;
                        POSPrinterBaseSetMapModeResponse res = new POSPrinterBaseSetMapModeResponse();
                        device.MapMode = (Microsoft.PointOfService.MapMode)Convert[req.MapMode];
                        result.Add(res);
                    }

                    // MapMode プロパティ (Getter)
                    else if (obj is POSPrinterBaseGetMapMode)
                    {
                        POSPrinterBaseGetMapMode req = (POSPrinterBaseGetMapMode)obj;
                        POSPrinterBaseGetMapModeResponse res = new POSPrinterBaseGetMapModeResponse();
                        res.GetMapModeResult = (MapMode)Convert[device.MapMode];
                        res.GetMapModeResultSpecified = true;
                        result.Add(res);
                    }
                }
            }
            catch (Exception e)
            {
                // 実行失敗
                try
                {
                    device.ClearOutput();
                }
                catch (Exception)
                {
                }
                response.POSPrinter.ARTSHeader.Response.ResponseCode = "Rejected";
                response.POSPrinter.ARTSHeader.Response.ResponseDescription.Value = e.Message;
            }

            // 応答メッセージに実行結果を追加する
            response.POSPrinter.POSPrinterBody = result.ToArray();

            // 応答メッセージを返す
            return response;
        }

        #endregion

        #region POSPrinter Dispose

        public void Dispose()
        {
            try
            {
                device.Close();
            }
            catch (Exception)
            {
            }
            device = null;
        }

        #endregion
    }
}